const fs = require('fs');
const path = require('path');
const os = require('os');
const { app } = require('electron');
const { PostHog } = require('posthog-node');

const { AnalyticsEventRegistry, FALLBACK_REGISTRY } = require('./analytics/registry');
const { EventConfigFetcher } = require('./analytics/eventConfigFetcher');

const DEFAULT_API_HOST = 'https://app.posthog.com';
const EVENT_QUEUE_LIMIT = 500;
const FLUSH_INTERVAL_MS = 10_000;
const EVENTS_CONFIG_FILENAME = 'posthog-events.json';
const DEFAULT_LOG_FILE = 'analytics-events.log';
const MAX_LOG_ENTRIES_DEFAULT = 1000;
const MAX_LOG_ENTRIES_DEBUG = 5000;

function isPlainObject(value) {
  return value != null && typeof value === 'object' && !Array.isArray(value);
}

class AnalyticsDiagnosticsLog {
  constructor() {
    this.active = false;
    this.retainAfterSend = false;
    this.filePath = null;
    this.entries = [];
    this.sequence = 0;
  }

  configure({ enabled = false, retainAfterSend = false, logFileName = DEFAULT_LOG_FILE } = {}) {
    this.active = Boolean(enabled);
    this.retainAfterSend = Boolean(retainAfterSend);
    this.sequence = 0;

    if (!this.active) {
      this.entries = [];
      this.filePath = null;
      return;
    }

    this.filePath = resolveLogFilePath(logFileName);
    this.entries = [];

    try {
      const payload = fs.readFileSync(this.filePath, 'utf8');
      const parsed = JSON.parse(payload);
      if (Array.isArray(parsed)) {
        this.entries = parsed;
      }
    } catch (error) {
      if (error && error.code !== 'ENOENT') {
        // Silently handle log read errors in production
        if (process.env.NODE_ENV === 'development' || !app.isPackaged) {
          console.warn(
            '[analytics-log] Failed to read diagnostics log:',
            error && error.message ? error.message : error
          );
        }
      }
    }

    this.trimEntries();
    this.persist();
  }

  persist() {
    if (!this.active || !this.filePath) {
      return;
    }
    try {
      fs.mkdirSync(path.dirname(this.filePath), { recursive: true });
      fs.writeFileSync(this.filePath, `${JSON.stringify(this.entries, null, 2)}\n`, 'utf8');
    } catch (error) {
      // Silently handle log write errors in production
      if (process.env.NODE_ENV === 'development' || !app.isPackaged) {
        console.warn(
          '[analytics-log] Failed to write diagnostics log:',
          error && error.message ? error.message : error
        );
      }
    }
  }

  trimEntries() {
    if (!this.active) {
      return;
    }
    const limit = this.retainAfterSend ? MAX_LOG_ENTRIES_DEBUG : MAX_LOG_ENTRIES_DEFAULT;

    if (this.entries.length <= limit) {
      return;
    }

    if (!this.retainAfterSend) {
      const remaining = [];
      for (const entry of this.entries) {
        if (entry.status !== 'sent') {
          remaining.push(entry);
        }
      }
      this.entries = remaining;
    }

    while (this.entries.length > limit) {
      this.entries.shift();
    }
  }

  recordAttempt({
    eventName,
    evaluation,
    properties,
    metadata
  }) {
    if (!this.active) {
      return null;
    }
    const id = `${Date.now()}-${process.pid}-${++this.sequence}`;
    const entry = {
      id,
      event: eventName,
      attemptedAt: new Date().toISOString(),
      status: evaluation.allow ? 'queued' : 'blocked',
      decision: evaluation.allow ? 'allowed' : 'blocked',
      reason: evaluation.reason || null,
      droppedProperties: evaluation.droppedProperties,
      properties,
      registryVersion: metadata.registryVersion,
      registryUpdatedAt: metadata.registryUpdatedAt || null,
      source: metadata.source || null,
      distinctId: metadata.distinctId || null
    };
    this.entries.push(entry);
    this.trimEntries();
    this.persist();
    return id;
  }

  updateStatus(id, status, { reason = null } = {}) {
    if (!this.active || !id) {
      return;
    }
    const index = this.entries.findIndex((entry) => entry.id === id);
    if (index === -1) {
      return;
    }
    const entry = this.entries[index];
    entry.status = status;
    entry.reason = reason || entry.reason;
    entry.updatedAt = new Date().toISOString();
    if (!this.retainAfterSend && status === 'sent') {
      this.entries.splice(index, 1);
    }
    this.trimEntries();
    this.persist();
  }
}

const diagnosticsLog = new AnalyticsDiagnosticsLog();

let cachedConfig = null;
let cachedConfigPath = null;
let eventsConfigPath = null;

const eventRegistry = new AnalyticsEventRegistry({ logger: console });
let registryBootstrapPromise = null;
let registryReady = false;
const registryListeners = new Set();

const eventQueue = [];
let analyticsRuntime = {
  enabled: false,
  distinctId: null,
  config: null,
  debug: {
    enabled: false,
    retainAfterSend: false,
    logFile: DEFAULT_LOG_FILE
  }
};

let posthogClient = null;
let flushTimer = null;

function resolveCandidatePaths() {
  const cwd = process.cwd();
  const appRoot = path.resolve(__dirname, '..', '..');
  const resourcesPath = process.resourcesPath || null;

  const candidates = [
    path.join(cwd, 'config', 'posthog.json'),
    path.join(appRoot, 'config', 'posthog.json')
  ];

  if (resourcesPath) {
    candidates.push(path.join(resourcesPath, 'config', 'posthog.json'));
  }

  return candidates;
}

function resolveEventConfigPaths() {
  const cwd = process.cwd();
  const appRoot = path.resolve(__dirname, '..', '..');
  const resourcesPath = process.resourcesPath || null;

  const candidates = [
    path.join(cwd, 'config', EVENTS_CONFIG_FILENAME),
    path.join(appRoot, 'config', EVENTS_CONFIG_FILENAME)
  ];

  if (resourcesPath) {
    candidates.push(path.join(resourcesPath, 'config', EVENTS_CONFIG_FILENAME));
  }

  return candidates;
}

function parseAnalyticsDebugConfig(payload) {
  const defaults = {
    enabled: false,
    retainAfterSend: false,
    logFile: DEFAULT_LOG_FILE
  };
  if (!isPlainObject(payload)) {
    return defaults;
  }
  const enabled = payload.enabled === true;
  const retainAfterSend = payload.retainAfterSend === true;
  const logFile =
    typeof payload.logFile === 'string' && payload.logFile.trim().length > 0
      ? payload.logFile.trim()
      : DEFAULT_LOG_FILE;
  return {
    enabled,
    retainAfterSend,
    logFile
  };
}

function readConfigFile() {
  const candidates = resolveCandidatePaths();

  for (const candidate of candidates) {
    try {
      if (fs.existsSync(candidate)) {
        const payload = JSON.parse(fs.readFileSync(candidate, 'utf8'));
        const projectApiKey = typeof payload.projectApiKey === 'string' ? payload.projectApiKey.trim() : '';
        const apiHost =
          typeof payload.apiHost === 'string' && payload.apiHost.trim().length > 0
            ? payload.apiHost.trim()
            : DEFAULT_API_HOST;
        const analyticsDebug = parseAnalyticsDebugConfig(payload.analyticsDebug);

        if (projectApiKey) {
          cachedConfig = { projectApiKey, apiHost, analyticsDebug };
          cachedConfigPath = candidate;
          return cachedConfig;
        }
      }
    } catch (error) {
      // Only log config parse errors in development
      if (process.env.NODE_ENV === 'development' || !app.isPackaged) {
        console.warn('Failed to parse PostHog config:', candidate, error && error.message ? error.message : error);
      }
    }
  }

  cachedConfig = null;
  cachedConfigPath = null;
  return null;
}

function getPosthogConfig(options = {}) {
  if (options.refresh || !cachedConfig) {
    return readConfigFile();
  }
  return cachedConfig;
}

function getPosthogConfigPath() {
  if (!cachedConfigPath) {
    readConfigFile();
  }
  return cachedConfigPath;
}

function resolveLogFilePath(fileName = DEFAULT_LOG_FILE) {
  const safeName = typeof fileName === 'string' && fileName.trim().length > 0 ? fileName.trim() : DEFAULT_LOG_FILE;
  let baseDir = process.cwd();
  try {
    if (app && typeof app.getPath === 'function') {
      baseDir = app.getPath('userData');
    }
  } catch {
    baseDir = process.cwd();
  }
  if (path.isAbsolute(safeName)) {
    return safeName;
  }
  return path.join(baseDir, 'logs', safeName);
}

function notifyRegistryListeners() {
  const snapshot = eventRegistry.getSnapshot();
  registryListeners.forEach((listener) => {
    try {
      listener(snapshot);
    } catch (error) {
      // Only log listener errors in development
      if (process.env.NODE_ENV === 'development' || !app.isPackaged) {
        console.warn('[analytics-registry] Listener error:', error && error.message ? error.message : error);
      }
    }
  });
}

function refreshDiagnosticsLogging() {
  diagnosticsLog.configure({
    enabled: analyticsRuntime.debug.enabled || analyticsRuntime.debug.retainAfterSend,
    retainAfterSend: analyticsRuntime.debug.retainAfterSend,
    logFileName: analyticsRuntime.debug.logFile
  });
}

function handleFetcherLifecycle(event) {
  if (!event || typeof event !== 'object') {
    return;
  }
  const attempt = Number.isFinite(event.attempt) ? event.attempt : 0;
  if (event.stage === 'error') {
    const reason =
      event.error && event.error.message
        ? event.error.message
        : event.statusCode
          ? `http_${event.statusCode}`
          : 'unknown_error';
    captureMainEvent('event_schema_download_failed', {
      reason,
      attempt,
      status_code: event.statusCode || null
    });
    if (event.error && event.error.name === 'SyntaxError') {
      captureMainEvent('event_schema_parse_failed', {
        reason,
        attempt
      });
    }
    return;
  }
  if (event.stage === 'not_modified') {
    captureMainEvent('event_schema_download_etag_same', {
      attempt,
      elapsed_ms: Number.isFinite(event.durationMs) ? event.durationMs : null
    });
    return;
  }
  if (event.stage === 'success') {
    captureMainEvent('event_schema_download_success', {
      attempt,
      elapsed_ms: Number.isFinite(event.durationMs) ? event.durationMs : null
    });
    if (event.updated) {
      refreshDiagnosticsLogging();
      captureMainEvent('new_schema_fetched', {
        attempt,
        elapsed_ms: Number.isFinite(event.durationMs) ? event.durationMs : null
      });
    }
  }
}

const eventConfigFetcher = new EventConfigFetcher({
  registry: eventRegistry,
  resolveConfigPaths: resolveEventConfigPaths,
  fallbackRemote: { ...FALLBACK_REGISTRY.remote },
  configFileName: EVENTS_CONFIG_FILENAME,
  logger: console,
  onConfigApplied: ({ path }) => {
    if (path) {
      eventsConfigPath = path;
    }
    notifyRegistryListeners();
  },
  onFetchLifecycle: handleFetcherLifecycle
});

async function bootstrapEventRegistry({ force = false } = {}) {
  if (force) {
    await eventConfigFetcher.fetchLatest({ force: true, reason: 'manual_force', block: true });
    registryReady = true;
    return eventRegistry.getSnapshot();
  }

  if (!registryReady) {
    if (!registryBootstrapPromise) {
      registryBootstrapPromise = eventConfigFetcher
        .bootstrap()
        .catch((error) => {
          // Only log bootstrap errors in development
          if (process.env.NODE_ENV === 'development' || !app.isPackaged) {
            console.warn('[analytics-registry] Bootstrap failed:', error && error.message ? error.message : error);
          }
          throw error;
        })
        .finally(() => {
          registryBootstrapPromise = null;
        });
    }

    await registryBootstrapPromise.catch(() => {});
    registryReady = true;
  }

  const snapshot = eventRegistry.getSnapshot();
  notifyRegistryListeners();
  return snapshot;
}

function ensureRegistryBootstrap() {
  return bootstrapEventRegistry().catch((error) => {
    // Only log bootstrap errors in development
    if (process.env.NODE_ENV === 'development' || !app.isPackaged) {
      console.warn('[analytics-registry] Bootstrap failed:', error && error.message ? error.message : error);
    }
    return eventRegistry.getSnapshot();
  });
}

function startFlushTimer() {
  if (flushTimer) {
    return;
  }
  flushTimer = setInterval(() => {
    flushQueuedEvents().catch((error) => {
      // Only log flush errors in development
      if (process.env.NODE_ENV === 'development' || !app.isPackaged) {
        console.warn('PostHog flush failed:', error && error.message ? error.message : error);
      }
    });
  }, FLUSH_INTERVAL_MS);
  if (typeof flushTimer.unref === 'function') {
    flushTimer.unref();
  }
}

function stopFlushTimer() {
  if (flushTimer) {
    clearInterval(flushTimer);
    flushTimer = null;
  }
}

async function flushQueuedEvents() {
  if (!posthogClient || eventQueue.length === 0) {
    return;
  }

  const batch = eventQueue.splice(0, eventQueue.length);
  const sentIds = [];

  for (const item of batch) {
    try {
      const properties = { ...(item.properties || {}) };
      if (!properties.source) {
        properties.source = item.source || 'main';
      }
      posthogClient.capture({
        distinctId: item.distinctId || analyticsRuntime.distinctId || undefined,
        event: item.event,
        properties,
        timestamp: item.timestamp ? new Date(item.timestamp) : undefined,
        groups: item.groups
      });
      if (item.diagnosticsId) {
        sentIds.push(item.diagnosticsId);
      }
    } catch (error) {
      // Only log capture errors in development
      if (process.env.NODE_ENV === 'development' || !app.isPackaged) {
        console.warn('Failed to capture analytics event:', error && error.message ? error.message : error);
      }
      if (item.diagnosticsId) {
        diagnosticsLog.updateStatus(item.diagnosticsId, 'failed', {
          reason: error && error.message ? error.message : 'capture_failed'
        });
      }
    }
  }

  let flushSucceeded = true;
  try {
    if (typeof posthogClient.flushAsync === 'function') {
      await posthogClient.flushAsync();
    }
  } catch (error) {
    flushSucceeded = false;
    // Only log flush errors in development
    if (process.env.NODE_ENV === 'development' || !app.isPackaged) {
      console.warn('PostHog flush encountered an error:', error && error.message ? error.message : error);
    }
  }

  if (!flushSucceeded) {
    for (let i = batch.length - 1; i >= 0; i -= 1) {
      eventQueue.unshift(batch[i]);
    }
    return;
  }

  sentIds.forEach((id) => {
    diagnosticsLog.updateStatus(id, 'sent');
  });
}

function shutdownAnalytics() {
  stopFlushTimer();
  if (posthogClient) {
    try {
      posthogClient.shutdown();
    } catch (error) {
      // Only log shutdown errors in development
      if (process.env.NODE_ENV === 'development' || !app.isPackaged) {
        console.warn('Error shutting down analytics client:', error && error.message ? error.message : error);
      }
    }
    posthogClient = null;
  }
  if (eventConfigFetcher) {
    eventConfigFetcher.stop();
  }
}

function applyAnalyticsRuntime({ analyticsSettings, config }) {
  const hasConfig = Boolean(config && config.projectApiKey);
  const enabled = Boolean(analyticsSettings && analyticsSettings.optedIn && hasConfig);
  const debugSettings = isPlainObject(config && config.analyticsDebug)
    ? {
        enabled: Boolean(config.analyticsDebug.enabled),
        retainAfterSend: Boolean(config.analyticsDebug.retainAfterSend),
        logFile:
          typeof config.analyticsDebug.logFile === 'string' && config.analyticsDebug.logFile.trim().length > 0
            ? config.analyticsDebug.logFile.trim()
            : DEFAULT_LOG_FILE
      }
    : {
        enabled: false,
        retainAfterSend: false,
        logFile: DEFAULT_LOG_FILE
      };

  diagnosticsLog.configure({
    enabled: debugSettings.enabled || debugSettings.retainAfterSend,
    retainAfterSend: debugSettings.retainAfterSend,
    logFileName: debugSettings.logFile
  });

  analyticsRuntime = {
    enabled,
    distinctId: enabled && analyticsSettings && analyticsSettings.userId ? analyticsSettings.userId : null,
    config: hasConfig ? config : null,
    debug: debugSettings
  };

  shutdownAnalytics();

  if (enabled && analyticsRuntime.config) {
    posthogClient = new PostHog(analyticsRuntime.config.projectApiKey, {
      host: analyticsRuntime.config.apiHost || DEFAULT_API_HOST,
      flushAt: 1
    });
    startFlushTimer();
    flushQueuedEvents().catch((error) => {
      // Only log initial flush errors in development
      if (process.env.NODE_ENV === 'development' || !app.isPackaged) {
        console.warn('Initial PostHog flush failed:', error && error.message ? error.message : error);
      }
    });
  } else {
    eventQueue.length = 0;
  }
}

function queueAnalyticsEvent(event) {
  if (!event || typeof event !== 'object') {
    return { success: false, error: 'Invalid event payload.' };
  }

  if (!analyticsRuntime.enabled) {
    return { success: true, skipped: true };
  }

  const { event: eventName, properties } = event;
  if (typeof eventName !== 'string' || eventName.trim().length === 0) {
    return { success: false, error: 'Event name is required.' };
  }

  const sanitized = {
    event: eventName.trim(),
    distinctId:
      typeof event.distinctId === 'string' && event.distinctId.trim().length > 0
        ? event.distinctId.trim()
        : analyticsRuntime.distinctId,
    properties: properties && typeof properties === 'object' ? { ...properties } : {},
    timestamp: event.timestamp || new Date().toISOString(),
    source: typeof event.source === 'string' ? event.source : 'renderer',
    groups: event.groups && typeof event.groups === 'object' ? { ...event.groups } : undefined
  };

  const registrySnapshot = eventRegistry.getSnapshot();
  const evaluation = eventRegistry.evaluateEvent(sanitized.event, sanitized.properties);
  const diagnosticsMetadata = {
    registryVersion: registrySnapshot.version,
    registryUpdatedAt: registrySnapshot.metadata ? registrySnapshot.metadata.loadedAt : null,
    source: sanitized.source,
    distinctId: sanitized.distinctId
  };

  if (!evaluation.allow) {
    const diagnosticsId = diagnosticsLog.recordAttempt({
      eventName: sanitized.event,
      evaluation,
      properties: evaluation.properties,
      metadata: diagnosticsMetadata
    });
    diagnosticsLog.updateStatus(diagnosticsId, 'blocked', { reason: evaluation.reason || 'event_not_allowed' });
    return {
      success: true,
      dropped: true,
      reason: evaluation.reason || 'event_not_allowed',
      registryVersion: registrySnapshot.version,
      registryUpdatedAt: diagnosticsMetadata.registryUpdatedAt,
      diagnosticsId: diagnosticsId || null
    };
  }

  sanitized.properties = {
    ...evaluation.properties,
    registry_version: registrySnapshot.version
  };

  if (analyticsRuntime.debug.enabled || analyticsRuntime.debug.retainAfterSend) {
    sanitized.properties.analytics_debug_mode = 1;
  }

  const diagnosticsId = diagnosticsLog.recordAttempt({
    eventName: sanitized.event,
    evaluation,
    properties: sanitized.properties,
    metadata: diagnosticsMetadata
  });

  sanitized.diagnosticsId = diagnosticsId;

  if (eventQueue.length >= EVENT_QUEUE_LIMIT) {
    const dropped = eventQueue.shift();
    if (dropped && dropped.diagnosticsId) {
      diagnosticsLog.updateStatus(dropped.diagnosticsId, 'dropped', { reason: 'queue_overflow' });
    }
  }

  eventQueue.push(sanitized);

  if (posthogClient) {
    flushQueuedEvents().catch((error) => {
      // Only log flush errors in development
      if (process.env.NODE_ENV === 'development' || !app.isPackaged) {
        console.warn('PostHog flush failed after enqueue:', error && error.message ? error.message : error);
      }
    });
  }

  return {
    success: true,
    diagnosticsId,
    registryVersion: registrySnapshot.version,
    registryUpdatedAt: diagnosticsMetadata.registryUpdatedAt
  };
}

function captureMainEvent(eventName, properties = {}, options = {}) {
  return queueAnalyticsEvent({
    event: eventName,
    properties,
    distinctId: options.distinctId,
    source: options.source || 'main',
    groups: options.groups,
    timestamp: options.timestamp
  });
}

function getRegistrySnapshot() {
  return eventRegistry.getSnapshot();
}

function onRegistryUpdated(listener) {
  if (typeof listener !== 'function') {
    return () => {};
  }
  registryListeners.add(listener);
  return () => registryListeners.delete(listener);
}

async function refreshEventRegistry({ force = true } = {}) {
  const result = await eventConfigFetcher.fetchLatest({
    force,
    reason: force ? 'manual_force' : 'manual',
    block: true
  });
  return {
    success: Boolean(result && result.success),
    updated: Boolean(result && result.updated),
    skipped: Boolean(result && result.skipped),
    error: result && result.error ? result.error.message || String(result.error) : null
  };
}

module.exports = {
  DEFAULT_API_HOST,
  getPosthogConfig,
  getPosthogConfigPath,
  applyAnalyticsRuntime,
  queueAnalyticsEvent,
  captureMainEvent,
  flushQueuedEvents,
  shutdownAnalytics,
  bootstrapEventRegistry: ensureRegistryBootstrap,
  getRegistrySnapshot,
  onRegistryUpdated,
  refreshEventRegistry,
  EVENTS_CONFIG_FILENAME,
  resolveEventConfigPaths,
  FALLBACK_REGISTRY
};

